#!/usr/bin/perl
# Dmitry Maksimov dima_dm@hotmail.com
# ./$0 cisco 172.16.0.1 75 5 public 5 lab-cisco
use Time::localtime;
use Net::SNMP;

$hostname = shift;		# Switch hostname in Zabbix
$hostip   = shift;		# IP of the switch
$ptpct    = shift;		# Port trigger will fire at this % load
				#     = 0 for no port triggers
$mem_ptpct= shift;		# Mem trigger will fire at this %
$cstring  = shift;		# Switch SNMP community string
$slow	  = shift;		# Delay factor for all items
@groups   = @ARGV;		# Host groups in Zabbix

unless (defined($hostip)) {
        print "Usage: $0 HOSTNAME IP [PORTLOAD] [MEMLOAD] [COMMUNITY] [MINUTES] [GROUPS]\n";
        print "Host name and IP are required parameters\n";
        print "Example: $0 localhost 127.0.0.1 75 5 public 5 lab-cisco\n";
        exit;
}

$hostname="Template_".$hostname;
unless (defined($ptpct))	{$ptpct   = 75};
unless (defined($mem_ptpct))	{$mem_ptpct = 5};
unless (defined($cstring))	{$cstring = 'public'};
unless (defined($slow))		{$slow	  = 5};
unless (defined(@groups))	{@groups  =('Cisco')};

print "\nHost:\t\t$hostname\nIP:\t\t$hostip\nPercent:\t$ptpct\nComm string:\t$cstring\n";
print "Slow:\t\t$slow\nGroups:\t\t$groups[0]\n";
#print "OK?";
#$ok = <>;
#chomp $ok;
#unless ($ok eq 'Y') {exit};

sub get_dt() {
	
	my $yr = localtime->year() - 100;
	my $mn = localtime->mon() + 1;
	my $dy = localtime->mday();
	my $hr = localtime->hour();
	my $mi = localtime->min();
	$dt = sprintf("%2.2d.%2.2d.%2.2d",$dy,$mn,$yr);
	$tm = sprintf("%2.2d.%2.2d",$hr,$mi);
	$dtm = sprintf("%2.2d%2.2d%2.2d%2.2d%2.2d",$yr,$mn,$dy,$hr,$mi);	
}

sub start_snmp() {
	
   ($session, $error) = Net::SNMP->session(
                           -hostname      => $hostip,
                           -port          => 161,
                           -version       => 2,
                           -community     => $cstring
                        );
   if (!defined($session)) {
      printf("ERROR: %s.\n", $error);
      exit 1;
   }
	
}

sub numerically { $a <=> $b;}

sub get_ifTable() {

	$ifindex = '.1.3.6.1.2.1.2.2.1.1';
	$ifAlias  = '.1.3.6.1.2.1.2.2.1.2';
	$ifdesc  = '.1.3.6.1.2.1.31.1.1.1.18';
	$iftype  = '.1.3.6.1.2.1.2.2.1.3';
	$ifspeed = '.1.3.6.1.2.1.2.2.1.5';
	start_snmp();
	my @columns = ($ifindex, $ifdesc,$iftype,$ifspeed,$ifAlias);
	$iftable = $session->get_entries(-columns => \@columns);
	unless (defined($iftable)) {
		printf("ERROR: %s.\n", $session->error);
		exit 1;
	}
	foreach $k (keys %$iftable) {
		print "$k\n";
	}
	$session->close();	
}

sub get_ports() {

	$ifNumber = '.1.3.6.1.2.1.2.1.0';
	$result = $session->get_request(
		-varbindlist => [$ifNumber]
	);

	$numports = $result->{$ifNumber};
	print "Num Ports: $numports\n";
	$np = 0;
	$in = 0;
	@ports = ();
	foreach $k (keys %$iftable) {
		unless($k =~ /^$ifindex/) {next};
		$r = $$iftable{$k};
		$np++;
		print "Port Index $np:$r\n";
		push @ports, $r;
	}
	@ports = sort numerically @ports;
	$max_port = $ports[$#ports];
	$len_port = int(log($max_port)/log(10)) + 1;
	print "Max Port: $max_port  $len_port\n";
}

sub get_fan() {
	$ciscoEnvMonFanState='.1.3.6.1.4.1.9.9.13.1.4.1.3';
	my $k;
	my @fan;
	my $num=0;
	start_snmp();
	my @columns = ($ciscoEnvMonFanState);
	$iffan = $session->get_entries(-columns => \@columns);
	unless (defined($iffan)) {
                printf("Fan not fountd. ERROR: %s.\n", $session->error);
        }
	foreach $k (keys %$iffan) {
		$num++;
                print "FAN_$num $k\n";
		push @fan, $k;
        }
        $session->close();
	return @fan;
}

get_dt();
open XML, ">$hostname.xml";

print XML <<EOF1;
<?xml version="1.0"?>
<zabbix_export version="1.0" date="$dt" time="$tm">
	<hosts>
		<host name="$hostname">
			<useip>1</useip>
			<ip>$hostip</ip>
			<port>10050</port>
			<groups>
EOF1
foreach $group (@groups) {print XML "\t\t\t\t<group>$group</group>\n"};
print XML <<EOF2;
			</groups>
			<items>
				<item type="4" key="sysUpTimeInstance" value_type="3">
                                        <description>sysUpTime</description>
                                        <delay>180*$slow</delay>
                                        <history>7</history>
                                        <trends>365</trends>
					<multiplier>1</multiplier>
					<formula>0.01</formula>
                                        <units>sec</units>
                                        <snmp_community>$cstring</snmp_community>
                                        <snmp_oid>1.3.6.1.2.1.1.3.0</snmp_oid>
                                        <snmp_port>161</snmp_port>
                                </item>
				<item type="4" key="ipForwDatagrams" value_type="3">
					<description>ipForwDatagrams - number of packets to forward</description>
					<delay>60*$slow</delay>
					<history>7</history>
					<trends>365</trends>
					<formula>1</formula>
					<units>Packet/sec</units>
					<delta>1</delta>
					<snmp_community>$cstring</snmp_community>
					<snmp_oid>1.3.6.1.2.1.4.6.0</snmp_oid>
					<snmp_port>161</snmp_port>
				</item>
				<item type="4" key="ipOutNoRoutes" value_type="3">
					<description>ipOutNoRoutes - discard caused no-route</description>
					<delay>60*$slow</delay>
					<history>7</history>
					<trends>365</trends>
					<formula>1</formula>
					<units>Packet/sec</units>
					<delta>1</delta>
					<snmp_community>$cstring</snmp_community>
					<snmp_oid>1.3.6.1.2.1.4.12.0</snmp_oid>
					<snmp_port>161</snmp_port>
				</item>
				<item type="4" key="ipInDiscards" value_type="3">
					<description>ipInDiscards - discards on input interfaces</description>
					<delay>60*$slow</delay>
					<history>7</history>
					<trends>365</trends>
					<formula>1</formula>
					<units>Packet/sec</units>
					<delta>1</delta>
					<snmp_community>$cstring</snmp_community>
					<snmp_oid>1.3.6.1.2.1.4.8.0</snmp_oid>
					<snmp_port>161</snmp_port>
				</item>
				<item type="4" key="ipOutDiscards" value_type="3">
					<description>ipOutDiscards - discards on output interfaces</description>
					<delay>60*$slow</delay>
					<history>7</history>
					<trends>365</trends>
					<formula>1</formula>
					<units>Packet/sec</units>
					<delta>1</delta>
					<snmp_community>$cstring</snmp_community>
					<snmp_oid>1.3.6.1.2.1.4.11.0</snmp_oid>
					<snmp_port>161</snmp_port>
				</item>
				<item type="4" key="Cpu" value_type="3">
					<description>% CPU</description>
					<delay>60*$slow</delay>
					<history>7</history>
					<trends>365</trends>
					<formula>1</formula>
					<snmp_community>$cstring</snmp_community>
					<snmp_oid>.1.3.6.1.4.1.9.2.1.56.0</snmp_oid>
					<snmp_port>161</snmp_port>
				</item>
				<item type="4" key="processorRam" value_type="3">
					<description>Memory Total</description>
					<delay>3600*$slow</delay>
					<history>7</history>
					<trends>365</trends>
					<formula>1</formula>
					<units>Byte</units>
					<snmp_community>$cstring</snmp_community>
					<snmp_oid>.1.3.6.1.4.1.9.3.6.6.0</snmp_oid>
					<snmp_port>161</snmp_port>
				</item>

				<item type="4" key="ciscoMemoryPoolLargestFree" value_type="3">
					<description>Memory Pool Largest Free</description>
					<delay>300*$slow</delay>
					<history>7</history>
					<trends>365</trends>
					<formula>1</formula>
					<units>Byte</units>
					<snmp_community>$cstring</snmp_community>
					<snmp_oid>.1.3.6.1.4.1.9.9.48.1.1.1.7.1</snmp_oid>
					<snmp_port>161</snmp_port>
				</item>

				<item type="4" key="ciscoMemoryPoolFree" value_type="3">
					<description>Memory Free</description>
					<delay>300*$slow</delay>
					<history>7</history>
					<trends>365</trends>
					<formula>1</formula>
					<units>Byte</units>
					<snmp_community>$cstring</snmp_community>
					<snmp_oid>.1.3.6.1.4.1.9.9.48.1.1.1.6.1</snmp_oid>
					<snmp_port>161</snmp_port>
                        	</item>
				
                                <item type="4" key="ciscoEnvMonTemperatureStatusValue" value_type="3">
                                        <description>Temperature in C</description>
                                        <delay>600*$slow</delay>
                                        <history>7</history>
                                        <trends>365</trends>
                                        <formula>1</formula>
                                        <units>C</units>
                                        <snmp_community>$cstring</snmp_community>
                                        <snmp_oid>.1.3.6.1.4.1.9.9.13.1.3.1.3.1</snmp_oid>
                                        <snmp_port>161</snmp_port>
                                </item>
				
                                <item type="4" key="CiscoEnvMonState" value_type="3">
                                        <description>Temperature State</description>
                                        <delay>600*$slow</delay>
                                        <history>7</history>
                                        <trends>365</trends>
                                        <formula>1</formula>
                                        <units></units>
                                        <snmp_community>$cstring</snmp_community>
                                        <snmp_oid>.1.3.6.1.4.1.9.9.13.1.3.1.6.1</snmp_oid>
                                        <snmp_port>161</snmp_port>
                                </item> 


				<item type="3" key="icmpping" value_type="3">
					<description>Simple_Ping</description>
					<delay>60</delay>
					<history>1</history>
					<trends>365</trends>
					<formula>1</formula>
					<snmp_community>public</snmp_community>
					<snmp_oid>interfaces.ifTable.ifEntry.ifInOctets.1</snmp_oid>
					<snmp_port>161</snmp_port>
				</item>
EOF2
@fan=get_fan();
$num=0;
foreach $i (@fan)
{
$num++;
print XML "\t\t\t\t<item type=\"4\" key=\"FAN_$num\" value_type=\"3\">\n";
print XML "\t\t\t\t\t<description>FAN $num status</description>\n";
print XML "\t\t\t\t\t<delay>720*$slow</delay>\n";
print XML "\t\t\t\t\t<history>7</history>\n";
print XML "\t\t\t\t\t<trends>365</trends>\n";
print XML "\t\t\t\t\t<snmp_community>$cstring</snmp_community>\n";
print XML "\t\t\t\t\t<snmp_oid>$i</snmp_oid>\n";
print XML "\t\t\t\t\t<snmp_port>161</snmp_port>\n";
print XML "\t\t\t\t</item>\n";
} 
get_ifTable();
get_ports();

foreach $port (@ports) {
$type = $$iftable{$iftype.'.'.$port};
if ($type ==   1) {next};
$portdesc = $$iftable{$ifdesc.'.'.$port};
$portdesc=~s/&/&amp;/g;
$portalias=$$iftable{$ifAlias.'.'.$port};
$portkey = $portalias;
$portkey =~ s#[^a-zA-Z0-9_]#_#g;
$sport = sprintf ("%${len_port}.${len_port}d",$port);
print XML "\t\t\t\t<item type=\"4\" key=\"Port_$portkey\_Description\" value_type=\"1\">\n";
print XML "\t\t\t\t\t<description>Port $sport $portalias Description $portdesc</description>\n";
print XML "\t\t\t\t\t<delay>86400*$slow</delay>\n";
print XML "\t\t\t\t\t<history>7</history>\n";
print XML "\t\t\t\t\t<trends>365</trends>\n";
print XML "\t\t\t\t\t<formula>1</formula>\n";
print XML "\t\t\t\t\t<snmp_community>$cstring</snmp_community>\n";
print XML "\t\t\t\t\t<snmp_oid>.1.3.6.1.2.1.2.2.1.2.$port</snmp_oid>\n";
print XML "\t\t\t\t\t<snmp_port>161</snmp_port>\n";
print XML "\t\t\t\t</item>\n";

print XML "\t\t\t\t<item type=\"4\" key=\"Port_$portkey\_InOctets\" value_type=\"3\">\n";
print XML "\t\t\t\t\t<description>Port $sport $portalias InOctets $portdesc</description>\n";
print XML "\t\t\t\t\t<delay>60*$slow</delay>\n";
print XML "\t\t\t\t\t<history>7</history>\n";
print XML "\t\t\t\t\t<trends>365</trends>\n";
print XML "\t\t\t\t\t<units>bps</units>\n";
print XML "\t\t\t\t\t<formula>8</formula>\n";
print XML "\t\t\t\t\t<multiplier>1</multiplier>\n";
print XML "\t\t\t\t\t<delta>1</delta>\n";
print XML "\t\t\t\t\t<snmp_community>$cstring</snmp_community>\n";
print XML "\t\t\t\t\t<snmp_oid>.1.3.6.1.2.1.2.2.1.10.$port</snmp_oid>\n";
print XML "\t\t\t\t\t<snmp_port>161</snmp_port>\n";
print XML "\t\t\t\t</item>\n";

print XML "\t\t\t\t<item type=\"4\" key=\"Port_$portkey\_InUcastPkts\" value_type=\"3\">\n";
print XML "\t\t\t\t\t<description>Port $sport $portalias InUcastPkts $portdesc</description>\n";
print XML "\t\t\t\t\t<delay>60*$slow</delay>\n";
print XML "\t\t\t\t\t<history>7</history>\n";
print XML "\t\t\t\t\t<trends>365</trends>\n";
print XML "\t\t\t\t\t<units>pps</units>\n";
print XML "\t\t\t\t\t<delta>1</delta>\n";
print XML "\t\t\t\t\t<snmp_community>$cstring</snmp_community>\n";
print XML "\t\t\t\t\t<snmp_oid>.1.3.6.1.2.1.2.2.1.11.$port</snmp_oid>\n";
print XML "\t\t\t\t\t<snmp_port>161</snmp_port>\n";
print XML "\t\t\t\t</item>\n";

print XML "\t\t\t\t<item type=\"4\" key=\"Port_$portkey\_InNUcastPkts\" value_type=\"3\">\n";
print XML "\t\t\t\t\t<description>Port $sport $portalias InNUcastPkts $portdesc</description>\n";
print XML "\t\t\t\t\t<delay>60*$slow</delay>\n";
print XML "\t\t\t\t\t<history>7</history>\n";
print XML "\t\t\t\t\t<trends>365</trends>\n";
print XML "\t\t\t\t\t<units>pps</units>\n";
print XML "\t\t\t\t\t<delta>1</delta>\n";
print XML "\t\t\t\t\t<snmp_community>$cstring</snmp_community>\n";
print XML "\t\t\t\t\t<snmp_oid>.1.3.6.1.2.1.2.2.1.12.$port</snmp_oid>\n";
print XML "\t\t\t\t\t<snmp_port>161</snmp_port>\n";
print XML "\t\t\t\t</item>\n";

print XML "\t\t\t\t<item type=\"4\" key=\"Port_$portkey\_OutUcastPkts\" value_type=\"3\">\n";
print XML "\t\t\t\t\t<description>Port $sport $portalias OutUcastPkts $portdesc</description>\n";
print XML "\t\t\t\t\t<delay>60*$slow</delay>\n";
print XML "\t\t\t\t\t<history>7</history>\n";
print XML "\t\t\t\t\t<trends>365</trends>\n";
print XML "\t\t\t\t\t<units>pps</units>\n";
print XML "\t\t\t\t\t<delta>1</delta>\n";
print XML "\t\t\t\t\t<snmp_community>$cstring</snmp_community>\n";
print XML "\t\t\t\t\t<snmp_oid>.1.3.6.1.2.1.2.2.1.17.$port</snmp_oid>\n";
print XML "\t\t\t\t\t<snmp_port>161</snmp_port>\n";
print XML "\t\t\t\t</item>\n";

print XML "\t\t\t\t<item type=\"4\" key=\"Port_$portkey\_OutNUcastPkts\" value_type=\"3\">\n";
print XML "\t\t\t\t\t<description>Port $sport $portalias OutNUcastPkts $portdesc</description>\n";
print XML "\t\t\t\t\t<delay>60*$slow</delay>\n";
print XML "\t\t\t\t\t<history>7</history>\n";
print XML "\t\t\t\t\t<trends>365</trends>\n";
print XML "\t\t\t\t\t<units>pps</units>\n";
print XML "\t\t\t\t\t<delta>1</delta>\n";
print XML "\t\t\t\t\t<snmp_community>$cstring</snmp_community>\n";
print XML "\t\t\t\t\t<snmp_oid>.1.3.6.1.2.1.2.2.1.18.$port</snmp_oid>\n";
print XML "\t\t\t\t\t<snmp_port>161</snmp_port>\n";
print XML "\t\t\t\t</item>\n";

print XML "\t\t\t\t<item type=\"4\" key=\"Port_$portkey\_InErrors\" value_type=\"3\">\n";
print XML "\t\t\t\t\t<description>Port $sport $portalias InErrors $portdesc</description>\n";
print XML "\t\t\t\t\t<delay>600*$slow</delay>\n";
print XML "\t\t\t\t\t<history>7</history>\n";
print XML "\t\t\t\t\t<trends>365</trends>\n";
print XML "\t\t\t\t\t<delta>2</delta>\n";
print XML "\t\t\t\t\t<snmp_community>$cstring</snmp_community>\n";
print XML "\t\t\t\t\t<snmp_oid>.1.3.6.1.2.1.2.2.1.14.$port</snmp_oid>\n";
print XML "\t\t\t\t\t<snmp_port>161</snmp_port>\n";
print XML "\t\t\t\t</item>\n";

print XML "\t\t\t\t<item type=\"4\" key=\"Port_$portkey\_InCRC\" value_type=\"3\">\n";
print XML "\t\t\t\t\t<description>Port $sport $portalias InCRC $portdesc</description>\n";
print XML "\t\t\t\t\t<delay>600*$slow</delay>\n";
print XML "\t\t\t\t\t<history>7</history>\n";
print XML "\t\t\t\t\t<trends>365</trends>\n";
print XML "\t\t\t\t\t<delta>2</delta>\n";
print XML "\t\t\t\t\t<snmp_community>$cstring</snmp_community>\n";
print XML "\t\t\t\t\t<snmp_oid>.1.3.6.1.4.1.9.2.2.1.1.12.$port</snmp_oid>\n";
print XML "\t\t\t\t\t<snmp_port>161</snmp_port>\n";
print XML "\t\t\t\t</item>\n";

print XML "\t\t\t\t<item type=\"4\" key=\"Port_$portkey\_OutOctets\" value_type=\"3\">\n";
print XML "\t\t\t\t\t<description>Port $sport $portalias OutOctets $portdesc</description>\n";
print XML "\t\t\t\t\t<delay>60*$slow</delay>\n";
print XML "\t\t\t\t\t<history>7</history>\n";
print XML "\t\t\t\t\t<trends>365</trends>\n";
print XML "\t\t\t\t\t<units>bps</units>\n";
print XML "\t\t\t\t\t<delta>1</delta>\n";
print XML "\t\t\t\t\t<formula>8</formula>\n";
print XML "\t\t\t\t\t<multiplier>1</multiplier>\n";
print XML "\t\t\t\t\t<snmp_community>$cstring</snmp_community>\n";
print XML "\t\t\t\t\t<snmp_oid>.1.3.6.1.2.1.2.2.1.16.$port</snmp_oid>\n";
print XML "\t\t\t\t\t<snmp_port>161</snmp_port>\n";
print XML "\t\t\t\t</item>\n";

print XML "\t\t\t\t<item type=\"4\" key=\"Port_$portkey\_OutErrors\" value_type=\"3\">\n";
print XML "\t\t\t\t\t<description>Port $sport $portalias OutErrors $portdesc</description>\n";
print XML "\t\t\t\t\t<delay>600*$slow</delay>\n";
print XML "\t\t\t\t\t<history>7</history>\n";
print XML "\t\t\t\t\t<trends>365</trends>\n";
print XML "\t\t\t\t\t<delta>2</delta>\n";
print XML "\t\t\t\t\t<snmp_community>$cstring</snmp_community>\n";
print XML "\t\t\t\t\t<snmp_oid>.1.3.6.1.2.1.2.2.1.20.$port</snmp_oid>\n";
print XML "\t\t\t\t\t<snmp_port>161</snmp_port>\n";
print XML "\t\t\t\t</item>\n";

print XML "\t\t\t\t<item type=\"4\" key=\"Port_$portkey\_Collisions\" value_type=\"3\">\n";
print XML "\t\t\t\t\t<description>Port $sport $portalias Collisions $portdesc</description>\n";
print XML "\t\t\t\t\t<delay>600*$slow</delay>\n";
print XML "\t\t\t\t\t<history>7</history>\n";
print XML "\t\t\t\t\t<trends>365</trends>\n";
print XML "\t\t\t\t\t<delta>2</delta>\n";
print XML "\t\t\t\t\t<snmp_community>$cstring</snmp_community>\n";
print XML "\t\t\t\t\t<snmp_oid>.1.3.6.1.4.1.9.2.2.1.1.25.$port</snmp_oid>\n";
print XML "\t\t\t\t\t<snmp_port>161</snmp_port>\n";
print XML "\t\t\t\t</item>\n";

print XML "\t\t\t\t<item type=\"4\" key=\"Port_$portkey\_Speed\" value_type=\"3\">\n";
print XML "\t\t\t\t\t<description>Port $sport $portalias Speed $portdesc</description>\n";
print XML "\t\t\t\t\t<delay>600*$slow</delay>\n";
print XML "\t\t\t\t\t<history>7</history>\n";
print XML "\t\t\t\t\t<trends>365</trends>\n";
print XML "\t\t\t\t\t<units>bps</units>\n";
print XML "\t\t\t\t\t<formula>1</formula>\n";
print XML "\t\t\t\t\t<snmp_community>$cstring</snmp_community>\n";
print XML "\t\t\t\t\t<snmp_oid>.1.3.6.1.2.1.2.2.1.5.$port</snmp_oid>\n";
print XML "\t\t\t\t\t<snmp_port>161</snmp_port>\n";
print XML "\t\t\t\t</item>\n";

print XML "\t\t\t\t<item type=\"4\" key=\"Port_$portkey\_OperStatus\" value_type=\"3\">\n";
print XML "\t\t\t\t\t<description>Port $sport $portalias OperStatus $portdesc</description>\n";
print XML "\t\t\t\t\t<delay>600*$slow</delay>\n";
print XML "\t\t\t\t\t<history>7</history>\n";
print XML "\t\t\t\t\t<trends>365</trends>\n";
print XML "\t\t\t\t\t<formula>1</formula>\n";
print XML "\t\t\t\t\t<valuemap>Port Status</valuemap>\n";
print XML "\t\t\t\t\t<snmp_community>$cstring</snmp_community>\n";
print XML "\t\t\t\t\t<snmp_oid>.1.3.6.1.2.1.2.2.1.8.$port</snmp_oid>\n";
print XML "\t\t\t\t\t<snmp_port>161</snmp_port>\n";
print XML "\t\t\t\t</item>\n";

print XML "\t\t\t\t<item type=\"4\" key=\"Port_$portkey\_AdminStatus\" value_type=\"3\">\n";
print XML "\t\t\t\t\t<description>Port $sport $portalias AdminStatus $portdesc</description>\n";
print XML "\t\t\t\t\t<delay>600*$slow</delay>\n";
print XML "\t\t\t\t\t<history>7</history>\n";
print XML "\t\t\t\t\t<trends>365</trends>\n";
print XML "\t\t\t\t\t<formula>1</formula>\n";
print XML "\t\t\t\t\t<valuemap>Port Status</valuemap>\n";
print XML "\t\t\t\t\t<snmp_community>$cstring</snmp_community>\n";
print XML "\t\t\t\t\t<snmp_oid>.1.3.6.1.2.1.2.2.1.7.$port</snmp_oid>\n";
print XML "\t\t\t\t\t<snmp_port>161</snmp_port>\n";
print XML "\t\t\t\t</item>\n";
}
print XML "\t\t\t</items>\n";
print XML "\t\t\t<triggers>\n";
$num=0;
foreach $i (@fan)
{
$num++;
	print XML "\t\t\t\t<trigger>\n";
        print XML "\t\t\t\t\t<description>FAN $num failed</description>\n";
        print XML "\t\t\t\t\t<expression>{{HOSTNAME}:FAN_$num.last(0)}#1</expression>\n";
        print XML "\t\t\t\t\t<priority>2</priority>\n";
        print XML "\t\t\t\t\t<status>0</status>\n";
        print XML "\t\t\t\t\t<comments>
        FAN_$num Failed
        </comments>\n";
        print XML "\t\t\t\t</trigger>\n";
}
$fact = $ptpct/100;
foreach $port (@ports) {
	$type = $$iftable{$iftype.'.'.$port};
	if ($type ==   1) {next};
	$sport = sprintf ("%3.3d",$port);
	$portdesc = $$iftable{$ifdesc.'.'.$port};
	$portdesc=~s/&/&amp;/g;
	$speed = $$iftable{$ifspeed.'.'.$port};
	$cspeed = $ptpct*$speed/100;
	$portalias=$$iftable{$ifAlias.'.'.$port};
$portkey = $portalias;
$portkey =~ s#[^a-zA-Z0-9_]#_#g;
	
   if ($ptpct != 0) {
	print XML "\t\t\t\t<trigger>\n";
	print XML "\t\t\t\t\t<description>[Port $portalias InOctets] $sport $portdesc InOctets Threshold Exceeded</description>\n";
	print XML "\t\t\t\t\t<expression>{{HOSTNAME}:Port_$portkey\_InOctets.min(#4)}&gt;$fact*{{HOSTNAME}:Port_$portkey\_Speed.max(#2)} &amp; {{HOSTNAME}:Port_$portkey\_Speed.max(#2)}#0</expression>\n";
	print XML "\t\t\t\t\t<priority>4</priority>\n";
	print XML "\t\t\t\t\t<status>0</status>\n";
	print XML "\t\t\t\t\t<comments>
	Port speed $speed bps
	Threshold = $ptpct\%
	</comments>\n";
	print XML "\t\t\t\t</trigger>\n";

	print XML "\t\t\t\t<trigger>\n";
        print XML "\t\t\t\t\t<description>[Port $portalias InErrors] $sport $portdesc InErrors on Interface </description>\n";
        print XML "\t\t\t\t\t<expression>{{HOSTNAME}:Port_$portkey\_InErrors.last(0)}&gt;0</expression>\n";
        print XML "\t\t\t\t\t<priority>1</priority>\n";
	print XML "\t\t\t\t\t<status>0</status>\n";
        print XML "\t\t\t\t\t<comments>
        InErrors on interface
        </comments>\n";
        print XML "\t\t\t\t</trigger>\n";

        print XML "\t\t\t\t<trigger>\n";
        print XML "\t\t\t\t\t<description>[Port $portalias InCRC] $sport $portdesc InCRC error on Interface </description>\n";
        print XML "\t\t\t\t\t<expression>{{HOSTNAME}:Port_$portkey\_InCRC.last(0)}&gt;0</expression>\n";
        print XML "\t\t\t\t\t<priority>1</priority>\n";
	print XML "\t\t\t\t\t<status>0</status>\n";
        print XML "\t\t\t\t\t<comments>
        InCRC error on interface
        </comments>\n";
        print XML "\t\t\t\t</trigger>\n";

	print XML "\t\t\t\t<trigger>\n";
	print XML "\t\t\t\t\t<description>[Port $portalias OutOctets] $sport $portdesc OutOctets Threshold Exceeded</description>\n";
	print XML "\t\t\t\t\t<expression>{{HOSTNAME}:Port_$portkey\_OutOctets.min(#4)}&gt;$fact*{{HOSTNAME}:Port_$portkey\_Speed.max(#2)} &amp; {{HOSTNAME}:Port_$portkey\_Speed.max(#2)}#0</expression>\n";
	print XML "\t\t\t\t\t<priority>4</priority>\n";
	print XML "\t\t\t\t\t<status>0</status>\n";
	print XML "\t\t\t\t\t<comments>Port speed $speed bps</comments>\n";
	print XML "\t\t\t\t</trigger>\n";


        print XML "\t\t\t\t<trigger>\n";
        print XML "\t\t\t\t\t<description>[Port $portalias OutErrors] $sport $portdesc OutErrors on Interface </description>\n";
        print XML "\t\t\t\t\t<expression>{{HOSTNAME}:Port_$portkey\_OutErrors.last(0)}&gt;0</expression>\n";
        print XML "\t\t\t\t\t<priority>1</priority>\n";
	print XML "\t\t\t\t\t<status>0</status>\n";
        print XML "\t\t\t\t\t<comments>
        OutErrors on interface
        </comments>\n";
        print XML "\t\t\t\t</trigger>\n";

        print XML "\t\t\t\t<trigger>\n";
        print XML "\t\t\t\t\t<description>[Port $portalias Collisions] $sport $portdesc Collisions on Interface </description>\n";
        print XML "\t\t\t\t\t<expression>{{HOSTNAME}:Port_$portkey\_Collisions.last(0)}&gt;0</expression>\n";
        print XML "\t\t\t\t\t<priority>1</priority>\n";
	print XML "\t\t\t\t\t<status>0</status>\n";
        print XML "\t\t\t\t\t<comments>
        Collisions on interface
        </comments>\n";
        print XML "\t\t\t\t</trigger>\n";


	print XML "\t\t\t\t<trigger>\n";
	print XML "\t\t\t\t\t<description>[Port $portalias AdminStatus] $sport $portdesc administrative status has changed</description>\n";
	print XML "\t\t\t\t\t<expression>{{HOSTNAME}:Port_$portkey\_AdminStatus.delta(#2)}#0</expression>\n";
	print XML "\t\t\t\t\t<priority>1</priority>\n";
	print XML "\t\t\t\t\t<status>0</status>\n";
	print XML "\t\t\t\t</trigger>\n";
   }	
	print XML "\t\t\t\t<trigger>\n";
	print XML "\t\t\t\t\t<description>[Port $portalias OperStatus] $sport $portdesc status has changed</description>\n";
	print XML "\t\t\t\t\t<expression>{{HOSTNAME}:Port_$portkey\_OperStatus.delta(#2)}#0</expression>\n";
	print XML "\t\t\t\t\t<priority>1</priority>\n";
	print XML "\t\t\t\t\t<status>0</status>\n";
	print XML "\t\t\t\t</trigger>\n";
}
$mem_fact=$mem_ptpct/100;
print XML <<EOF3;
				<trigger>
					<description>{HOSTNAME} CPU Over $ptpct %</description>
					<expression>{{HOSTNAME}:Cpu.min(90)}&gt;$ptpct</expression>
					<priority>4</priority>
					<status>0</status>
					<comments>{HOSTNAME} has had a CPU Utilization above $ptpct % for 3 samples taken in the last 1 minute 30 seconds.</comments>
				</trigger>
				
				<trigger>
                                        <description>{HOSTNAME} SNMP Failed</description>
                                        <expression>{{HOSTNAME}:Cpu.nodata(14400)}=1</expression>
                                        <priority>4</priority>
					<status>0</status>
                                        <comments>{HOSTNAME} SNMP Failed. Check configuration, ACL.</comments>
                                </trigger>
				
				<trigger>
					<description>Memory Free &lt; $mem_ptpct %</description>
					<type>0</type>
					<expression>{{HOSTNAME}:ciscoMemoryPoolFree.min(10)}/{{HOSTNAME}:processorRam.last(0)}&lt; $mem_fact</expression>
					<url></url>
					<status>0</status>
					<priority>3</priority>
					<comments>Memory Free &lt; $mem_ptpct %</comments>
				</trigger>

                                <trigger>
                                        <description>{HOSTNAME} High Temperature </description>
                                        <expression>{{HOSTNAME}:CiscoEnvMonState.last(0)}&gt;1</expression>
                                        <priority>4</priority>
					<status>0</status>
                                        <comments>{HOSTNAME} High Temperature. Possible values:
1:normal
2:warning
3:critical
4:shutdown</comments>
                                </trigger>

				<trigger>
					<description>{HOSTNAME} is Unreachable</description>
					<expression>{{HOSTNAME}:icmpping.max(90)}=0</expression>
					<priority>4</priority>
					<status>0</status>
					<comments>Can not Ping {HOSTNAME} 3 times in last 1 minute 30</comments>
				</trigger>

				<trigger>
                                        <description>{HOSTNAME} reloaded</description>
                                        <expression>{{HOSTNAME}:sysUpTimeInstance.last(0)}&lt;720*$slow</expression>
                                        <priority>2</priority>
                                        <status>0</status>
                                        <comments>{HOSTNAME}  reloaded</comments>
                                </trigger>

EOF3

print XML "\t\t\t</triggers>\n";

print XML "\t\t\t<graphs>\n";
foreach $port (@ports) {
$type = $$iftable{$iftype.'.'.$port};
if ($type ==   1) {next};
$portdesc = $$iftable{$ifdesc.'.'.$port};
$portdesc=~s/&/&amp;/g;
$portalias=$$iftable{$ifAlias.'.'.$port};
$portkey = $portalias;
$portkey =~ s#[^a-zA-Z0-9_]#_#g;
$sport = sprintf ("%${len_port}.${len_port}d",$port);
print XML "\t\t\t\t<graph name=\"[Port $portalias] $sport $portdesc\" width=\"900\" height=\"400\">\n";
print XML <<EOF4;
					<yaxistype>0</yaxistype>
					<show_work_period>0</show_work_period>
					<show_triggers>0</show_triggers>
					<graphtype>0</graphtype>
					<ymin_type>1</ymin_type>
					<ymax_type>0</ymax_type>
					<ymin_item_key></ymin_item_key>
					<ymax_item_key></ymax_item_key>
					<yaxismin>0.0000</yaxismin>
					<yaxismax>100.0000</yaxismax>
					<show_legend>0</show_legend>
					<show_3d>0</show_3d>
					<percent_left>0.0000</percent_left>
					<percent_right>0.0000</percent_right>
					<graph_elements>
EOF4
print XML "\t\t\t\t\t\t<graph_element item=\"{HOSTNAME}:Port_$portkey\_OutOctets\">\n";
print XML <<EOF5;
							<drawtype>1</drawtype>
							<sortorder>0</sortorder>
							<color>009900</color>
							<yaxisside>0</yaxisside>
							<calc_fnc>4</calc_fnc>
							<type>0</type>
							<periods_cnt>5</periods_cnt>
						</graph_element>
EOF5
print XML "\t\t\t\t\t\t<graph_element item=\"{HOSTNAME}:Port_$portkey\_InOctets\">\n";
print XML <<EOF6;
							<drawtype>2</drawtype>
							<sortorder>0</sortorder>
							<color>000099</color>
							<yaxisside>0</yaxisside>
							<calc_fnc>4</calc_fnc>
							<type>0</type>
							<periods_cnt>5</periods_cnt>
						</graph_element>
				        </graph_elements>
					</graph>
EOF6
print XML "\t\t\t\t<graph name=\"[Port Packets $portalias] $sport $portdesc\" width=\"900\" height=\"400\">\n";
print XML <<EOF14;
                                        <yaxistype>0</yaxistype>
                                        <show_work_period>0</show_work_period>
                                        <show_triggers>0</show_triggers>
                                        <graphtype>0</graphtype>
                                        <ymin_type>1</ymin_type>
                                        <ymax_type>0</ymax_type>
                                        <ymin_item_key></ymin_item_key>
                                        <ymax_item_key></ymax_item_key>
                                        <yaxismin>0.0000</yaxismin>
                                        <yaxismax>100.0000</yaxismax>
                                        <show_legend>0</show_legend>
                                        <show_3d>0</show_3d>
                                        <percent_left>0.0000</percent_left>
                                        <percent_right>0.0000</percent_right>
                                        <graph_elements>
EOF14
print XML "\t\t\t\t\t\t<graph_element item=\"{HOSTNAME}:Port_$portkey\_InUcastPkts\">\n";
print XML <<EOF15;
                                                        <drawtype>2</drawtype>
                                                        <sortorder>0</sortorder>
                                                        <color>009900</color>
                                                        <yaxisside>0</yaxisside>
                                                        <calc_fnc>4</calc_fnc>
                                                        <type>0</type>
                                                        <periods_cnt>5</periods_cnt>
                                                </graph_element>
EOF15
print XML "\t\t\t\t\t\t<graph_element item=\"{HOSTNAME}:Port_$portkey\_OutUcastPkts\">\n";
print XML <<EOF16;
                                                        <drawtype>2</drawtype>
                                                        <sortorder>0</sortorder>
                                                        <color>000099</color>
                                                        <yaxisside>0</yaxisside>
                                                        <calc_fnc>4</calc_fnc>
                                                        <type>0</type>
                                                        <periods_cnt>5</periods_cnt>
                                                </graph_element>
EOF16
print XML "\t\t\t\t\t\t<graph_element item=\"{HOSTNAME}:Port_$portkey\_OutNUcastPkts\">\n";
print XML <<EOF17;
                                                        <drawtype>2</drawtype>
                                                        <sortorder>0</sortorder>
                                                        <color>880000</color>
                                                        <yaxisside>0</yaxisside>
                                                        <calc_fnc>4</calc_fnc>
                                                        <type>0</type>
                                                        <periods_cnt>5</periods_cnt>
                                                </graph_element>
EOF17
print XML "\t\t\t\t\t\t<graph_element item=\"{HOSTNAME}:Port_$portkey\_InNUcastPkts\">\n";
print XML <<EOF18;
                                                        <drawtype>2</drawtype>
                                                        <sortorder>0</sortorder>
                                                        <color>888800</color>
                                                        <yaxisside>0</yaxisside>
                                                        <calc_fnc>4</calc_fnc>
                                                        <type>0</type>
                                                        <periods_cnt>5</periods_cnt>
                                                </graph_element> 
					</graph_elements>
				</graph>
EOF18
}
print XML <<EOF7;
				<graph name="[RouterPerformance] - Total forwarded/dropped packets" width="900" height="400">
					<yaxistype>2</yaxistype>
					<show_work_period>0</show_work_period>
					<show_triggers>0</show_triggers>
					<graphtype>0</graphtype>
					<ymin_type>1</ymin_type>
					<ymax_type>0</ymax_type>
					<ymin_item_key></ymin_item_key>
					<ymax_item_key></ymax_item_key>
					<yaxismin>0.0000</yaxismin>
					<yaxismax>100.0000</yaxismax>
					<show_legend>0</show_legend>
					<show_3d>0</show_3d>
					<percent_left>0.0000</percent_left>
					<percent_right>0.0000</percent_right>
                                        <graph_elements>
						<graph_element item="{HOSTNAME}:ipForwDatagrams">
							<drawtype>5</drawtype>
							<sortorder>0</sortorder>
							<color>009900</color>
							<yaxisside>0</yaxisside>
							<calc_fnc>2</calc_fnc>
							<type>0</type>
							<periods_cnt>5</periods_cnt>
						</graph_element>
							<graph_element item="{HOSTNAME}:ipInDiscards">
							<drawtype>5</drawtype>
							<sortorder>0</sortorder>
							<color>990000</color>
							<yaxisside>0</yaxisside>
							<calc_fnc>2</calc_fnc>
							<type>0</type>
							<periods_cnt>5</periods_cnt>
						</graph_element>
                                                <graph_element item="{HOSTNAME}:ipOutDiscards">
							<drawtype>5</drawtype>
							<sortorder>0</sortorder>
							<color>990099</color>
							<yaxisside>0</yaxisside>
							<calc_fnc>2</calc_fnc>
							<type>0</type>
							<periods_cnt>5</periods_cnt>
						</graph_element>
						<graph_element item="{HOSTNAME}:ipOutNoRoutes">
							<drawtype>5</drawtype>
							<sortorder>0</sortorder>
							<color>FFFF00</color>
							<yaxisside>0</yaxisside>
							<calc_fnc>2</calc_fnc>
							<type>0</type>
							<periods_cnt>5</periods_cnt>
						</graph_element>
					</graph_elements>
				</graph>
                                <graph name="CPU LOAD %" width="900" height="400">
                                        <yaxistype>2</yaxistype>
                                        <show_work_period>0</show_work_period>
                                        <show_triggers>0</show_triggers>
					<graphtype>0</graphtype>
					<ymin_type>1</ymin_type>
					<ymax_type>0</ymax_type>
					<ymin_item_key></ymin_item_key>
					<ymax_item_key></ymax_item_key>
                                        <yaxismin>0.0000</yaxismin>
                                        <yaxismax>100.0000</yaxismax>
                                        <show_legend>0</show_legend>
                                        <show_3d>0</show_3d>
                                        <percent_left>0.0000</percent_left>
                                        <percent_right>0.0000</percent_right>
                                        <graph_elements>
                                                <graph_element item="{HOSTNAME}:Cpu">
                                                        <drawtype>0</drawtype>
                                                        <sortorder>0</sortorder>
                                                        <color>009900</color>
                                                        <yaxisside>0</yaxisside>
                                                        <calc_fnc>2</calc_fnc>
                                                        <type>0</type>
                                                        <periods_cnt>5</periods_cnt>
                                                </graph_element>
                                        </graph_elements>
                                </graph>

				<graph name="Memory" width="900" height="400">
					<yaxistype>2</yaxistype>
					<show_work_period>0</show_work_period>
					<show_triggers>1</show_triggers>
					<graphtype>0</graphtype>
					<ymin_type>1</ymin_type>
					<ymax_type>0</ymax_type>
					<ymin_item_key></ymin_item_key>
					<ymax_item_key></ymax_item_key>
					<yaxismin>0.0000</yaxismin>
					<yaxismax>100.0000</yaxismax>
					<show_legend>0</show_legend>
					<show_3d>0</show_3d>
					<percent_left>0.0000</percent_left>
					<percent_right>0.0000</percent_right>
					<graph_elements>
						<graph_element item="{HOSTNAME}:processorRam">
							<drawtype>0</drawtype>
							<sortorder>0</sortorder>
							<color>FF0000</color>
							<yaxisside>0</yaxisside>
							<calc_fnc>2</calc_fnc>
							<type>0</type>
							<periods_cnt>5</periods_cnt>
						</graph_element>
						<graph_element item="{HOSTNAME}:ciscoMemoryPoolFree">
							<drawtype>0</drawtype>
							<sortorder>0</sortorder>
							<color>009900</color>
							<yaxisside>0</yaxisside>
							<calc_fnc>2</calc_fnc>
							<type>0</type>
							<periods_cnt>5</periods_cnt>
						</graph_element>
						<graph_element item="{HOSTNAME}:ciscoMemoryPoolLargestFree">
							<drawtype>0</drawtype>
							<sortorder>0</sortorder>
							<color>0000CC</color>
							<yaxisside>0</yaxisside>
							<calc_fnc>2</calc_fnc>
							<type>0</type>
							<periods_cnt>5</periods_cnt>
						</graph_element>
					</graph_elements>
				</graph>

                               <graph name="Temperature" width="900" height="400">
                                        <yaxistype>2</yaxistype>
                                        <show_work_period>0</show_work_period>
                                        <show_triggers>0</show_triggers>
                                        <graphtype>0</graphtype>
					<ymin_type>1</ymin_type>
					<ymax_type>0</ymax_type>
					<ymin_item_key></ymin_item_key>
					<ymax_item_key></ymax_item_key>
                                        <yaxismin>0.0000</yaxismin>
                                        <yaxismax>100.0000</yaxismax>
                                        <show_legend>0</show_legend>
                                        <show_3d>0</show_3d>
                                        <percent_left>0.0000</percent_left>
                                        <percent_right>0.0000</percent_right>
                                        <graph_elements>
                                                <graph_element item="{HOSTNAME}:ciscoEnvMonTemperatureStatusValue">
                                                        <drawtype>0</drawtype>
                                                        <sortorder>0</sortorder>
                                                        <color>009900</color>
                                                        <yaxisside>0</yaxisside>
                                                        <calc_fnc>2</calc_fnc>
                                                        <type>0</type>
                                                        <periods_cnt>5</periods_cnt>
                                                </graph_element>
                                        </graph_elements>
                                </graph>

EOF7
print XML "\t\t\t</graphs>\n";
print XML "\t\t</host>\n";
print XML "\t</hosts>\n";
print XML "</zabbix_export>\n";
close XML;
exit;

